//
// $Id: fmcrtc.cc,v 1.6 2001/07/18 17:43:31 nishi Exp $
//
// Copyright (C) 2001 Shouhei Nishi.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of author may not be used to endorse or promote products
//    derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//


//#define VIDEO_MODE_TEST

#include "bochs.h"
#if BX_EMULATION_TOWNS
#define LOG_THIS bx_vga.
#include <math.h>
#ifdef VIDEO_MODE_TEST
#ifdef WIN32
#include <windows.h>
#define sleep(l) Sleep((l)*1000)
#define usleep(l) Sleep((l)/1000)
#endif
#endif

/* NOTES:
 * I take it data rotate is a true rotate with carry of bit 0 to bit 7.
 * support map mask (3c5 reg 02)
 */

// (mch)
#define VGA_TRACE_FEATURE

bx_vga_c bx_vga;
#if BX_USE_VGA_SMF
#define this (&bx_vga)
#endif

bx_vga_c::bx_vga_c(void)
{
  BX_VGA_THIS s.partial_updated = 0;
  BX_VGA_THIS s.all_updated = 0;
  BX_VGA_THIS s.dispmode_updated = 0;
  BX_VGA_THIS s.linfo[0].depth = 0;
  BX_VGA_THIS s.linfo[1].depth = 0;
  BX_VGA_THIS s.x_tilesize = X_TILESIZE;
  BX_VGA_THIS s.y_tilesize = Y_TILESIZE;
  BX_VGA_THIS setprefix("[VGA ]");
}


bx_vga_c::~bx_vga_c(void)
{
  // nothing for now
}


  void
bx_vga_c::init(bx_devices_c *d, bx_cmos_c *cmos)
{
  unsigned i;

  BX_VGA_THIS devices = d;

  static struct {
    Bit16u num;
    Boolean read;
    Boolean write;
  } video_ports[]={
    {0x0400,1,0},
    {0x0404,1,1},
    {0x0440,1,1},
    {0x0442,1,1},
    {0x0443,1,1},
    {0x0448,0,1},
    {0x044a,0,1},
    {0x044c,1,0},
    {0x0450,1,1},
    {0x0452,1,1},
    {0x0458,0,1},
    {0x045a,1,1},
    {0x045b,1,1},
    {0x05c8,1,0},
    {0x05ca,0,1},
    {0xff81,1,1},
    {0xfd90,1,1},
    {0xfd92,1,1},
    {0xfd94,1,1},
    {0xfd96,1,1},
    {0xfd98,1,1},
    {0xfd99,1,1},
    {0xfd9a,1,1},
    {0xfd9b,1,1},
    {0xfd9c,1,1},
    {0xfd9d,1,1},
    {0xfd9e,1,1},
    {0xfd9f,1,1},
    {0xfda0,1,1},
    {0,0,0}
  };

  i=0;
  while(video_ports[i].num!=0) {
    if(video_ports[i].read) {
      BX_VGA_THIS devices->register_io_read_handler(this, read_handler,
						    video_ports[i].num,
						    "video");
    }
    if(video_ports[i].write) {
      BX_VGA_THIS devices->register_io_write_handler(this, write_handler,
						     video_ports[i].num,
						     "video");
    }
    i++;
  }

  //memset(BX_VGA_THIS s.vga_memory, 0, sizeof(BX_VGA_THIS s.vga_memory));

  BX_VGA_THIS s.partial_updated = 0;
  BX_VGA_THIS s.all_updated = 0;
  BX_VGA_THIS s.dispmode_updated = 1;

  {
  /* ??? should redo this to pass X args */
  char *argv[1] = { "bochs" };
  bx_gui.init(1, &argv[0], BX_VGA_THIS s.x_tilesize, BX_VGA_THIS s.y_tilesize);
  }

  BX_INFO(("vga: interval=%lu\n", bx_options.vga_update_interval));
  bx_pc_system.register_timer(this, (bx_timer_handler_t)timer_handler,
     bx_options.vga_update_interval, 1, 1);
  BX_VGA_THIS vsync_timer_handle = bx_pc_system.register_timer(this,
							       (bx_timer_handler_t)vsync_timer_handler,
							       0,
							       1,
							       0);
  BX_VGA_THIS s.vsync_enable = 0;
  BX_VGA_THIS s.vsync_trigered = 0;
}


  // static IO port read callback handler
  // redirects to non-static class handler to avoid virtual functions

  Bit32u
bx_vga_c::read_handler(void *this_ptr, Bit32u address, unsigned io_len)
{
#if !BX_USE_VGA_SMF
  bx_vga_c *class_ptr = (bx_vga_c *) this_ptr;

  return( class_ptr->read(address, io_len) );
}


  Bit32u
bx_vga_c::read(Bit32u address, unsigned io_len)
{
#else
  UNUSED(this_ptr);
#endif  // !BX_USE_VGA_SMF
  Boolean saveMD;

#if defined(VGA_TRACE_FEATURE)
  Bit32u ret = 0;
#define RETURN(x) do { ret = (x); goto read_return; } while (0)
#else
#define RETURN return
#endif

#if !defined(VGA_TRACE_FEATURE)
    BX_DEBUG(("vga_io_read(%04x)!\n", (unsigned) address));
#endif

  if (address == 0xfd92 || address == 0xfd94) {
  } else if (address == 0x0442 || address == 0x45a) {
    if (io_len > 2)
      BX_PANIC(("vga: io read from address %08x, len=%u\n",
		(unsigned) address, (unsigned) io_len));
  } else {
    if (io_len > 1)
      BX_PANIC(("vga: io read from address %08x, len=%u\n",
		(unsigned) address, (unsigned) io_len));
  }

  switch (address) {
    case 0x0400:
      RETURN(0x00);

    case 0x0404:
      RETURN(BX_VGA_THIS devices->mem->MAIN_MEM ? 0x80 : 0);

    case 0x0442:
      if((BX_VGA_THIS s.CRTC.addr & 0x1f) == 30) BX_VGA_THIS calculate_hvsync();
      if(io_len==1) {
        RETURN(BX_VGA_THIS s.CRTC.reg[BX_VGA_THIS s.CRTC.addr & 0x1f] & 0xff);
      } else {
        RETURN(BX_VGA_THIS s.CRTC.reg[BX_VGA_THIS s.CRTC.addr & 0x1f]);
      }

    case 0x0443:
      if((BX_VGA_THIS s.CRTC.addr & 0x1f) == 30) BX_VGA_THIS calculate_hvsync();
      RETURN((BX_VGA_THIS s.CRTC.reg[BX_VGA_THIS s.CRTC.addr & 0x1f] >> 8) & 0xff);

    case 0x044c:
      saveMD=BX_VGA_THIS s.FMR_compat.DPMD;
      BX_VGA_THIS s.FMR_compat.DPMD=0;
      RETURN((saveMD ? 0x80 : 0) |
	     ((BX_VGA_THIS s.sprite.reg[1] & 0x80) != 0 ?
	      (rand()<RAND_MAX/2 ? 0x02 : 0) : 0) |
	     //((BX_VGA_THIS s.sprite.reg[1] & 0x80) != 0 ? 0x02 : 0) |
	     ((BX_VGA_THIS s.sprite.reg[6] & 0x10) != 0 ? 0x01 : 0));

    case 0x0452:
      RETURN(BX_VGA_THIS s.sprite.reg[BX_VGA_THIS s.sprite.addr & 0x7]);

    case 0x05c8:
      saveMD=BX_VGA_THIS s.FMR_compat.TVRAM_MD;
      BX_VGA_THIS s.FMR_compat.TVRAM_MD=0;
      RETURN(saveMD ? 0x80 : 0);

    case 0xff81:
      RETURN(mem_read(0xcff81));

    case 0xfd92:
      if(io_len == 4) {
	RETURN((BX_VGA_THIS pel_address(current_peltype()
					,BX_VGA_THIS s.pel.addr)->blue
		& 0xff) |
	       ((BX_VGA_THIS pel_address(current_peltype()
					 ,BX_VGA_THIS s.pel.addr)->red
		 << 16) & 0xff0000));
      } else {
	RETURN(BX_VGA_THIS pel_address(current_peltype()
				       ,BX_VGA_THIS s.pel.addr)->blue);
      }

    case 0xfd94:
      if(io_len == 4) {
	RETURN((BX_VGA_THIS pel_address(current_peltype()
					,BX_VGA_THIS s.pel.addr)->red
		& 0xff) |
	       ((BX_VGA_THIS pel_address(current_peltype()
					 ,BX_VGA_THIS s.pel.addr)->green
		 << 16) & 0xff0000));
      } else {
	RETURN(BX_VGA_THIS pel_address(current_peltype()
				       ,BX_VGA_THIS s.pel.addr)->red);
      }

    case 0xfd96:
      RETURN(BX_VGA_THIS pel_address(current_peltype()
				     ,BX_VGA_THIS s.pel.addr)->green);

    case 0xfd98:
    case 0xfd99:
    case 0xfd9a:
    case 0xfd9b:
    case 0xfd9c:
    case 0xfd9d:
    case 0xfd9e:
    case 0xfd9f:
      RETURN(BX_VGA_THIS s.FMR_compat.degipal[address-0xfd98] & 0xf);

    case 0xfda0:
      BX_VGA_THIS calculate_hvsync();;
      RETURN(BX_VGA_THIS s.FMR_compat.sub_status_reg);

    default:
      BX_PANIC(("*** io read from vga port %x\n", (unsigned) address));
      RETURN(0); /* keep compiler happy */
    }

#if defined(VGA_TRACE_FEATURE)
  read_return:
  if (bx_dbg.video)
    switch (io_len) {
    case 1:
      BX_INFO(("vga: 8-bit read to %04x = %02x\n", (unsigned)address, (unsigned)ret));
      break;
    case 2:
      BX_INFO(("vga: 16-bit read to %04x = %04x\n", (unsigned)address, (unsigned)ret));
      break;
    case 4:
      BX_INFO(("vga: 32-bit read to %04x = %08x\n", (unsigned)address, (unsigned)ret));
      break;
    default:
      BX_PANIC(("Weird VGA read size\n"));
    }
  return ret;
#endif
}
#if defined(VGA_TRACE_FEATURE)
#undef RETURN
#endif

  // static IO port write callback handler
  // redirects to non-static class handler to avoid virtual functions

  void
bx_vga_c::write_handler(void *this_ptr, Bit32u address, Bit32u value, unsigned io_len)
{
#if !BX_USE_VGA_SMF
  bx_vga_c *class_ptr = (bx_vga_c *) this_ptr;

  class_ptr->write(address, value, io_len, 0);
#else
  UNUSED(this_ptr);
  bx_vga.write(address, value, io_len, 0);
#endif
}

  void
bx_vga_c::write_handler_no_log(void *this_ptr, Bit32u address, Bit32u value, unsigned io_len)
{
#if !BX_USE_VGA_SMF
  bx_vga_c *class_ptr = (bx_vga_c *) this_ptr;

  class_ptr->write(address, value, io_len, 1);
#else
  UNUSED(this_ptr);
  bx_vga.write(address, value, io_len, 1);
#endif
}

  void
bx_vga_c::write(Bit32u address, Bit32u value, unsigned io_len, Boolean no_log)
{
#if defined(VGA_TRACE_FEATURE)
  if (!no_log && bx_dbg.video)
	switch (io_len) {
	      case 1:
		    BX_INFO(("vga: 8-bit write to %04x = %02x\n", (unsigned)address, (unsigned)value));
		    break;
	      case 2:
		    BX_INFO(("vga: 16-bit write to %04x = %04x\n", (unsigned)address, (unsigned)value));
		    break;
	      case 4:
		    BX_INFO(("vga: 32-bit write to %04x = %08x\n", (unsigned)address, (unsigned)value));
		    break;
	      default:
		    BX_PANIC(("Weird VGA write size\n"));
	}
#endif

#if !defined(VGA_TRACE_FEATURE)
  if (bx_dbg.video)
    BX_INFO(("vga_io_write(%04x)=%02x!\n", (unsigned) address,
      (unsigned) value));
#endif

  if (address == 0xfd92 || address == 0xfd94) {
  } else if (address == 0x0442 || address == 0x45a || address == 0x0440) {
    if (io_len > 2)
      BX_PANIC(("vga: io write to address %08x, len=%u\n",
		(unsigned) address, (unsigned) io_len));
  } else {
    if (io_len > 1)
      BX_PANIC(("vga: io write to address %08x, len=%u\n",
		(unsigned) address, (unsigned) io_len));
  }

  switch (address) {
    case 0x0404:
      BX_VGA_THIS devices->mem->MAIN_MEM = ((value & 0x80)!=0);
      return;

    case 0x0440:
      BX_VGA_THIS s.CRTC.addr=value;
      return;

    case 0x0442:
      if(io_len==1) {
	BX_VGA_THIS s.CRTC.reg[BX_VGA_THIS s.CRTC.addr & 0x1f] =
	  (BX_VGA_THIS s.CRTC.reg[BX_VGA_THIS s.CRTC.addr & 0x1f] & 0xff00) |
	  (value & 0x00ff);
      } else {
	BX_VGA_THIS s.CRTC.reg[BX_VGA_THIS s.CRTC.addr & 0x1f] = value;
      }
      BX_VGA_THIS s.dispmode_updated = 1;
      if((BX_VGA_THIS s.CRTC.addr & 0x1f) == 28) BX_VGA_THIS update();
      return;

    case 0x0443:
      BX_VGA_THIS s.CRTC.reg[BX_VGA_THIS s.CRTC.addr & 0x1f] =
	(BX_VGA_THIS s.CRTC.reg[BX_VGA_THIS s.CRTC.addr & 0x1f] & 0x00ff) |
	((value&0x00ff)<<8);
      BX_VGA_THIS s.dispmode_updated = 1;
      if((BX_VGA_THIS s.CRTC.addr & 0x1f) == 28) BX_VGA_THIS update();
      return;

    case 0x0448:
      BX_VGA_THIS s.misc_output.shifter_addr=value;
      return;

    case 0x044a:
      BX_VGA_THIS s.misc_output.shifter[BX_VGA_THIS s.misc_output.shifter_addr & 0x1] = value;
      BX_VGA_THIS s.dispmode_updated = 1;
      return;

    case 0x0450:
      BX_VGA_THIS s.sprite.addr=value;
      return;

    case 0x0452:
      BX_VGA_THIS s.sprite.reg[BX_VGA_THIS s.sprite.addr & 0x7] = value;
      return;

    case 0x0458:
      BX_VGA_THIS s.vram_ctrl.addr=value;
      return;

    case 0x045a:
      if(io_len==1) {
	BX_VGA_THIS s.vram_ctrl.mask[((BX_VGA_THIS s.vram_ctrl.addr&1)<<1)|0]=value;
      } else {
	BX_VGA_THIS s.vram_ctrl.mask[((BX_VGA_THIS s.vram_ctrl.addr&1)<<1)|0]=value;
	BX_VGA_THIS s.vram_ctrl.mask[((BX_VGA_THIS s.vram_ctrl.addr&1)<<1)|1]=value >> 8;
      }
      return;

    case 0x045b:
      BX_VGA_THIS s.vram_ctrl.mask[((BX_VGA_THIS s.vram_ctrl.addr&1)<<1)|1]=value;
      return;

    case 0x05ca:
      if(BX_VGA_THIS s.vsync_trigered) {
	BX_VGA_THIS s.vsync_trigered = 0;
	BX_VGA_THIS devices->pic->untrigger_irq(11);
      }

    case 0xff81:
      mem_write(0xcff81,value);
      return;

    case 0xfd90:
      BX_VGA_THIS s.pel.addr=value;
      return;

    case 0xfd92:
      BX_VGA_THIS pel_update(current_peltype(),
			     BX_VGA_THIS s.pel.addr,
			     (io_len == 4),(value >> 16) & 0xff,
			     0,0,
			     1,value);
      return;

    case 0xfd94:
      BX_VGA_THIS pel_update(current_peltype(),
			     BX_VGA_THIS s.pel.addr,
			     1,value,
			     (io_len == 4),(value >> 16) & 0xff,
			     0,0);
      return;

    case 0xfd96:
      BX_VGA_THIS pel_update(current_peltype(),
			     BX_VGA_THIS s.pel.addr,
			     0,0,
			     1,value,
			     0,0);
      return;

    case 0xfd98:
    case 0xfd99:
    case 0xfd9a:
    case 0xfd9b:
    case 0xfd9c:
    case 0xfd9d:
    case 0xfd9e:
    case 0xfd9f:
      BX_VGA_THIS s.FMR_compat.degipal[address-0xfd98] = value & 0xf;
      BX_VGA_THIS s.FMR_compat.DPMD = 1;
      break;

    case 0xfda0:
      BX_VGA_THIS s.misc_output.output_ctrl = value;
      BX_VGA_THIS s.dispmode_updated = 1;
      break;

    default:
      BX_PANIC(("vga: unsupported io write to port %x, val=%02x\n",
        (unsigned) address, (unsigned) value));
    }
}


  void
bx_vga_c::timer_handler(void *this_ptr)
{
  bx_vga_c *class_ptr = (bx_vga_c *) this_ptr;

  class_ptr->timer();
}

  void
bx_vga_c::timer(void)
{
  #ifdef VIDEO_MODE_TEST
  video_mode_test();
  BX_PANIC(("vga:video mode test is done.\n"));
  #endif
  update();
  bx_gui.flush();

}


  void
bx_vga_c::vsync_timer_handler(void *this_ptr)
{
  bx_vga_c *class_ptr = (bx_vga_c *) this_ptr;

  class_ptr->vsync_timer();
}

  void
bx_vga_c::vsync_timer(void)
{
  BX_VGA_THIS s.vsync_trigered_time = bx_pc_system.get_current_time();
  if(!BX_VGA_THIS s.vsync_trigered) {
    BX_VGA_THIS s.vsync_trigered = 1;
    BX_VGA_THIS devices->pic->trigger_irq(11);
  }
  BX_VGA_THIS s.field = !BX_VGA_THIS s.field;

  BX_VGA_THIS sprite_update();
}

  void
bx_vga_c::calculate_hvsync(void)
{
  int i;
  double vsync_time;
  double hsync_time;
  vsync_time = (bx_pc_system.get_current_time() -
		BX_VGA_THIS s.vsync_trigered_time) / 1000000.0;
  hsync_time = fmod(vsync_time,BX_VGA_THIS s.hsync_rate);
  BX_VGA_THIS s.CRTC.reg[30]                &= 0x00ff;
  BX_VGA_THIS s.FMR_compat.status_reg        = 0x10;
  BX_VGA_THIS s.FMR_compat.sub_status_reg    = 0x00;
  if(!BX_VGA_THIS s.vsync_enable) return;
  if(vsync_time < BX_VGA_THIS s.vsync_width) {
    BX_VGA_THIS s.CRTC.reg[30]              |= 0x0400;
    BX_VGA_THIS s.FMR_compat.status_reg     |=   0x04;
    BX_VGA_THIS s.FMR_compat.sub_status_reg |= 0x01;
  }
  if(hsync_time < BX_VGA_THIS s.hsync_width) {
    BX_VGA_THIS s.CRTC.reg[30]              |= 0x0200;
    BX_VGA_THIS s.FMR_compat.status_reg     |=   0x80;
    BX_VGA_THIS s.FMR_compat.sub_status_reg |= 0x02;
  }
  for(i=0;i<2;i++) {
    if(BX_VGA_THIS s.vdisp_top[i] <= vsync_time &&
       vsync_time <= BX_VGA_THIS s.vdisp_bottom[i]) {
      BX_VGA_THIS s.CRTC.reg[30]              |= 0x4000 << i;
    }
    if(BX_VGA_THIS s.hdisp_top[i] <= hsync_time &&
       hsync_time <= BX_VGA_THIS s.hdisp_bottom[i]) {
      BX_VGA_THIS s.CRTC.reg[30]              |= 0x1000 << i;
    }
  }
  if(BX_VGA_THIS s.field) {
    BX_VGA_THIS s.CRTC.reg[30]          |= 0x0800;
  }
}

  void
bx_vga_c::update(void)
{
  layer_info newinfo[2],templ;
  signed xfact,yfact,top,left,bottom,right;
  int i;
  Bit32u dispoff;
  unsigned zoomsave;
  int xti,yti,x,y;
  Bit32u addr;
  unsigned xparity;
  Bit32u pel;
  double  new_vsync_rate;
  double  new_hsync_rate;
  double  new_vsync_width;
  double  new_hsync_width;
  double  dotclk;
  Boolean new_vsync_enable;

  if(BX_VGA_THIS s.dispmode_updated) {
    BX_VGA_THIS s.dispmode_updated=0;
    newinfo[0].depth=newinfo[1].depth=0;
    BX_INFO(("vga: output_ctrl=%.2x\n",BX_VGA_THIS s.misc_output.output_ctrl));
    for(i=0;i<2;i++) {
      BX_INFO(("vga: shifter[%d]=%.2x\n",i,BX_VGA_THIS s.misc_output.shifter[i]));
    }
    for(i=0;i<32;i++) {
      BX_INFO(("vga: CRTC[%.2d]=%.2x\n",i,BX_VGA_THIS s.CRTC.reg[i]));
    }

    switch(BX_VGA_THIS s.CRTC.reg[29] & 0x3) {
    case 0x0:
      dotclk = 28.6363 * 1000000;
      break;
    case 0x1:
      dotclk = 24.5454 * 1000000;
      break;
    case 0x2:
      dotclk = 25.175  * 1000000;
      break;
    case 0x3:
    default:
      dotclk = 21.0525 * 1000000;
      break;
    }
    new_hsync_rate   = ((BX_VGA_THIS s.CRTC.reg[4]  & 0x7fe) + 2) / dotclk;
    new_hsync_width  =  (BX_VGA_THIS s.CRTC.reg[0]  & 0xfe)       / dotclk;
    new_vsync_rate   =  (BX_VGA_THIS s.CRTC.reg[8]  & 0x7ff)
      * new_hsync_rate / 2;
    new_vsync_width  = ((BX_VGA_THIS s.CRTC.reg[6] -
			 BX_VGA_THIS s.CRTC.reg[5]) & 0x1f)
      * new_hsync_rate / 2;
    new_vsync_enable =  (BX_VGA_THIS s.CRTC.reg[28] & 0x8000)     !=0;
    if(!new_vsync_enable) {
      bx_pc_system.deactivate_timer(BX_VGA_THIS vsync_timer_handle);
      BX_VGA_THIS s.vsync_enable = 0;
    } else {
      if(BX_VGA_THIS s.vsync_rate != new_vsync_rate ||
	 !BX_VGA_THIS s.vsync_enable) {
	BX_INFO(("set vsync rate = %g(%gHz)\n",new_vsync_rate,1/new_vsync_rate));
	BX_INFO(("set hsync rate = %g(%gKHz)\n",new_hsync_rate,0.001/new_hsync_rate));
	BX_INFO(("set dotclk     = %g(%gMHz)\n",dotclk,dotclk/1000000));
	bx_pc_system.activate_timer(BX_VGA_THIS vsync_timer_handle,
				    new_vsync_rate * 1000000,
				    1);
      }
      BX_VGA_THIS s.hsync_rate   = new_hsync_rate;
      BX_VGA_THIS s.hsync_width  = new_hsync_width;
      BX_VGA_THIS s.vsync_rate   = new_vsync_rate;
      BX_VGA_THIS s.vsync_width  = new_vsync_width;
      BX_VGA_THIS s.vsync_enable = 1;
    }
    for(i=0;i<2;i++) {
      BX_VGA_THIS s.vdisp_top[i]=((BX_VGA_THIS s.CRTC.reg[13+i*2]&0x7ff)
				  -(BX_VGA_THIS s.CRTC.reg[5]&0x1f))
	* new_hsync_rate / 2;
      BX_VGA_THIS s.vdisp_bottom[i]=((BX_VGA_THIS s.CRTC.reg[14+i*2]&0x7ff)
				     -(BX_VGA_THIS s.CRTC.reg[5]&0x1f))
	* new_hsync_rate / 2;
      BX_VGA_THIS s.hdisp_top[i]=(BX_VGA_THIS s.CRTC.reg[9+i*2]&0x7ff)/dotclk;
      BX_VGA_THIS s.hdisp_bottom[i]=(BX_VGA_THIS s.CRTC.reg[10+i*2]&0x7ff)/dotclk;
    }

    if((BX_VGA_THIS s.misc_output.shifter[1] & 0x04) != 0) {
      BX_INFO(("vga: YM=1 (not yet).\n"));
    }
    if((BX_VGA_THIS s.CRTC.reg[28]&0x8000)!=0) {
      switch(BX_VGA_THIS s.misc_output.shifter[0] & 0x1f) {
      case 0x0a:
	newinfo[0].depth     = 8;
	newinfo[0].layeraddr = 0x00000000;
	newinfo[0].layermask = 0x0007ffff;
	newinfo[0].paloffset = OFFSET_256COLOR_0;
	break;
      case 0x0f:
	newinfo[0].depth     = 16;
	newinfo[0].layeraddr = 0x00000000;
	newinfo[0].layermask = 0x0007ffff;
	newinfo[0].paloffset = OFFSET_HIGHCOLOR;
	break;
      case 0x15:
	newinfo[0].depth     = 4;
	newinfo[0].layeraddr = 0x00000000;
	newinfo[0].layermask = 0x0003ffff;
	newinfo[0].paloffset = OFFSET_16COLOR_0;
	newinfo[1].depth     = 4;
	newinfo[1].layeraddr = 0x00040000;
	newinfo[1].layermask = 0x0003ffff;
	newinfo[1].paloffset = OFFSET_16COLOR_1;
	break;
      case 0x17:
	newinfo[0].depth     = 16;
	newinfo[0].layeraddr = 0x00000000;
	newinfo[0].layermask = 0x0003ffff;
	newinfo[0].paloffset = OFFSET_HIGHCOLOR;
	newinfo[1].depth     = 4;
	newinfo[1].layeraddr = 0x00040000;
	newinfo[1].layermask = 0x0003ffff;
	newinfo[1].paloffset = OFFSET_16COLOR_1;
	break;
      case 0x1d:
	newinfo[0].depth     = 4;
	newinfo[0].layeraddr = 0x00000000;
	newinfo[0].layermask = 0x0003ffff;
	newinfo[0].paloffset = OFFSET_16COLOR_0;
	newinfo[1].depth     = 16;
	newinfo[1].layeraddr = 0x00040000;
	newinfo[1].layermask = 0x0003ffff;
	newinfo[1].paloffset = OFFSET_HIGHCOLOR;
	break;
      case 0x1f:
	newinfo[0].depth     = 16;
	newinfo[0].layeraddr = 0x00000000;
	newinfo[0].layermask = 0x0003ffff;
	newinfo[0].paloffset = OFFSET_HIGHCOLOR;
	newinfo[1].depth     = 16;
	newinfo[1].layeraddr = 0x00040000;
	newinfo[1].layermask = 0x0003ffff;
	newinfo[1].paloffset = OFFSET_HIGHCOLOR;
	break;
      default:
	BX_INFO(("vga: unsupported display mode = %02x\n", (unsigned)(BX_VGA_THIS s.misc_output.shifter[0] & 0x1f)));
	break;
      }
    }
  retry:
    if(new_hsync_rate>1.0/1000/20) {
      /* 15KHz */
      xfact=1;
      yfact=0;
    } else {
      /* 24, 31KHz */
      xfact=0;
      yfact=1;
    }
    top    = left  =  100000;
    bottom = right = -100000;
    if(new_hsync_rate>1.0/1000/20) {
      /* 15KHz */
      left = (0xe7 + 0x5e7 - DEFWIDTH15KHZ * 2) / 2 / (24.5454 * 1000000) * dotclk + 0.5;
      right = ((0xe7 + 0x5e7 + DEFWIDTH15KHZ * 2) / 2 - 1) / (24.5454 * 1000000) * dotclk + 0.5;
      top = (0x2a + 0x20a - DEFHEIGHT15KHZ) / 2;
      bottom = (0x2a + 0x20a + DEFHEIGHT15KHZ) / 2 - 1;
    } else if(new_hsync_rate>1.0/1000/27) {
      /* 24KHz */
      left = (0x9c + 0x31c - DEFWIDTH24KHZ) / 2 / (21.0525 * 1000000) * dotclk + 0.5;
      right = ((0x9c + 0x31c + DEFWIDTH24KHZ) / 2 - 1) / (21.0525 * 1000000) * dotclk + 0.5;
      top = (0x40 + 0x360 - DEFHEIGHT24KHZ * 2) / 2;
      bottom = (0x40 + 0x360 + DEFHEIGHT24KHZ * 2) / 2 - 1;
    } else {
      /* 31KHz */
      left = (0x8a + 0x30a - DEFWIDTH31KHZ) / 2 / (25.175 * 1000000) * dotclk + 0.5;
      right = ((0x8a + 0x30a + DEFWIDTH31KHZ) / 2 - 1) / (25.175 * 1000000) * dotclk + 0.5;
      top = (0x46 + 0x406 - DEFHEIGHT31KHZ * 2) / 2;
      bottom = (0x46 + 0x406 + DEFHEIGHT31KHZ * 2) / 2 - 1;
    }
    for(i=0;i<2;i++) {
      if(((BX_VGA_THIS s.misc_output.output_ctrl>>(2-i*2))&3)==0) {
	newinfo[i].depth     = 0;
      }
      if(newinfo[i].depth!=0) {
	if(newinfo[i].depth>16) {
	  BX_INFO(("vga: unsupported display mode = %02x\n", (unsigned)(BX_VGA_THIS s.misc_output.shifter[0] & 0x1f)));
	  newinfo[i].depth     = 0;
	  goto retry;
	}
	newinfo[i].top       = BX_VGA_THIS s.CRTC.reg[13+i*2]&0x7ff;
	newinfo[i].bottom    = (BX_VGA_THIS s.CRTC.reg[14+i*2]&0x7ff)-1;
	newinfo[i].left      = BX_VGA_THIS s.CRTC.reg[9+i*2]&0x7ff;
	newinfo[i].right     = (BX_VGA_THIS s.CRTC.reg[10+i*2]&0x7ff)-1;
	newinfo[i].xzoom2    = (((BX_VGA_THIS s.CRTC.reg[27]>>(i*8))&0xf)+1)*2;
	newinfo[i].yzoom     = ((BX_VGA_THIS s.CRTC.reg[27]>>(i*8+4))&0xf)+1;
	newinfo[i].displeft  = BX_VGA_THIS s.CRTC.reg[18+i*4]&0x7ff;
	newinfo[i].lineoff   = BX_VGA_THIS s.CRTC.reg[20+i*4];
	dispoff              = BX_VGA_THIS s.CRTC.reg[17+i*4];
	if(BX_VGA_THIS s.CRTC.reg[7]!=0) {
	  /* interlace */
	  if(BX_VGA_THIS s.CRTC.reg[19+i*4] ==
	     BX_VGA_THIS s.CRTC.reg[20+i*4]/2 &&
	     newinfo[i].yzoom == 1) {
	    newinfo[i].lineoff /= 2;
	  } else if(BX_VGA_THIS s.CRTC.reg[19+i*4]==0) {
	    newinfo[i].yzoom *= 2;
	  } else {
	    BX_INFO(("vga: unsupported interlace mode\n"));
	    newinfo[i].depth     = 0;
	    goto retry;
	  }
	} else {
	  /* non-interlace */
	  newinfo[i].yzoom *= 2;
	}
	if((BX_VGA_THIS s.misc_output.shifter[0] & 0x10)==0) {
	  /* 1 layer mode */
	  dispoff*=8;
	  newinfo[i].lineoff*=8;
	} else {
	  /* 2 layer mode */
	  dispoff*=4;
	  newinfo[i].lineoff*=4;
	}
	if((BX_VGA_THIS s.sprite.reg[6]&0x10) != 0 && i == 1) {
	  dispoff |= 0x20000;
	}
	if((BX_VGA_THIS s.CRTC.reg[28]&(0x10<<i))==0) {
	  /* spherical scroll */
	  newinfo[i].dispoffh  = dispoff & 0xfffffc00;
	  newinfo[i].dispoffl  = dispoff & 0x000003ff;
	  newinfo[i].vrollmask = 0x3ff;
	} else {
	  /* cylindrical scroll */
	  newinfo[i].dispoffh  = dispoff;
	  newinfo[i].dispoffl  = 0;
	  newinfo[i].vrollmask = 0xffffffff;
	}
	if(newinfo[i].top      >= newinfo[i].bottom ||
	   newinfo[i].left     >= newinfo[i].right ||
	   newinfo[i].displeft >  newinfo[i].left) {
	  BX_INFO(("vga: illegal display geometry\n"));
	  newinfo[i].depth     = 0;
	  goto retry;
	}
	if(newinfo[i].top<top) {
	  top=newinfo[i].top;
	}
	if(newinfo[i].bottom>bottom) {
	  bottom=newinfo[i].bottom;
	}
	if(newinfo[i].left<left) {
	  left=newinfo[i].left;
	}
	if(newinfo[i].right>right) {
	  right=newinfo[i].right;
	}
      }
    }
    if(newinfo[0].depth!=0 ||
       newinfo[1].depth!=0) {
      while(((bottom - top  + 1) >> yfact) > BX_MAX_YSIZE ||
	    ((right  - left + 1) >> xfact) > BX_MAX_XSIZE) {
	xfact++;
	yfact++;
      }
      if(((bottom - top  + 1) >> yfact) <= 0 ||
	 ((right  - left + 1) >> xfact) <= 0) {
	BX_INFO(("vga: illegal display geometry\n"));
	newinfo[0].depth=newinfo[1].depth=0;
	goto retry;
      }
      for(i=0;i<2;i++) {
	if(newinfo[i].depth!=0) {
	  newinfo[i].top       -= top;
	  newinfo[i].top      >>= yfact;
	  newinfo[i].bottom    -= top;
	  newinfo[i].bottom   >>= yfact;
	  newinfo[i].left      -= left;
	  newinfo[i].left     >>= xfact;
	  newinfo[i].right     -= left;
	  newinfo[i].right    >>= xfact;
	  zoomsave=newinfo[i].xzoom2;
	  newinfo[i].xzoom2   >>= xfact;
	  if((newinfo[i].xzoom2<<xfact)!=zoomsave) {
	    BX_INFO(("vga: illegal zoom factor\n"));
	    //newinfo[i].depth=0;
	    //goto retry;
	  }
	  zoomsave=newinfo[i].yzoom;
	  newinfo[i].yzoom    >>= yfact;
	  if((newinfo[i].yzoom<<yfact)!=zoomsave) {
	    BX_INFO(("vga: illegal zoom factor\n"));
	    //newinfo[i].depth=0;
	    //goto retry;
	  }
	  newinfo[i].displeft  -= left;
	  newinfo[i].displeft >>= xfact;
	}
      }
      if(newinfo[0].depth==0) {
	newinfo[0]=newinfo[1];
	newinfo[1].depth=0;
      }
      if(newinfo[1].depth!=0 &&
	 (BX_VGA_THIS s.misc_output.shifter[1]&1)==0) {
	/* reverse layer */
	templ=newinfo[0];
	newinfo[0]=newinfo[1];
	newinfo[1]=templ;
      }
      if(BX_VGA_THIS s.x_dispsize != ((right  - left + 1) >> xfact) ||
	 BX_VGA_THIS s.y_dispsize != ((bottom - top  + 1) >> yfact)) {
	BX_VGA_THIS s.x_dispsize = ((right  - left + 1) >> xfact);
	BX_VGA_THIS s.y_dispsize = ((bottom - top  + 1) >> yfact);
	bx_gui.dimension_update(BX_VGA_THIS s.x_dispsize,
				BX_VGA_THIS s.y_dispsize);
	bx_gui.flush();
	bx_gui.show_headerbar();
	BX_VGA_THIS s.all_updated = 1;
	BX_VGA_THIS s.dispmode_updated = 1;
	return;
      }
      for(i=0;i<2;i++) {
	if(newinfo[i].depth==0) {
	  BX_INFO(("vga : page %d is not displayed\n",i));
	} else {
	  BX_INFO(("vga : page %d :      depth : %d\n",i,newinfo[i].depth));
	  BX_INFO(("                      top : %d\n",  newinfo[i].top));
	  BX_INFO(("                   bottom : %d\n",  newinfo[i].bottom));
	  BX_INFO(("                     left : %d\n",  newinfo[i].left));
	  BX_INFO(("                    right : %d\n",  newinfo[i].right));
	  BX_INFO(("                   xzoom2 : %d\n",  newinfo[i].xzoom2));
	  BX_INFO(("                    yzoom : %d\n",  newinfo[i].yzoom));
	  BX_INFO(("                layeraddr : %.8x\n",newinfo[i].layeraddr));
	  BX_INFO(("                layermask : %.8x\n",newinfo[i].layermask));
	  BX_INFO(("                 displeft : %.8x\n",newinfo[i].displeft));
	  BX_INFO(("                  lineoff : %.8x\n",newinfo[i].lineoff));
	  BX_INFO(("                 dispoffh : %.8x\n",newinfo[i].dispoffh));
	  BX_INFO(("                 dispoffl : %.8x\n",newinfo[i].dispoffl));
	  BX_INFO(("                vrollmask : %.8x\n",newinfo[i].vrollmask));
	  BX_INFO(("                paloffset : %.8x\n",newinfo[i].paloffset));
	}
      }
      for(i=0;i<2;i++) {
	if(newinfo[i].depth             == 0 &&
	   BX_VGA_THIS s.linfo[i].depth == 0) {
	  /* don't need update */
	} else if(newinfo[i].depth     == BX_VGA_THIS s.linfo[i].depth     &&
		  newinfo[i].top       == BX_VGA_THIS s.linfo[i].top       &&
		  newinfo[i].bottom    == BX_VGA_THIS s.linfo[i].bottom    &&
		  newinfo[i].left      == BX_VGA_THIS s.linfo[i].left      &&
		  newinfo[i].right     == BX_VGA_THIS s.linfo[i].right     &&
		  newinfo[i].xzoom2    == BX_VGA_THIS s.linfo[i].xzoom2    &&
		  newinfo[i].yzoom     == BX_VGA_THIS s.linfo[i].yzoom     &&
		  newinfo[i].layeraddr == BX_VGA_THIS s.linfo[i].layeraddr &&
		  newinfo[i].layermask == BX_VGA_THIS s.linfo[i].layermask &&
		  newinfo[i].lineoff   == BX_VGA_THIS s.linfo[i].lineoff   &&
		  newinfo[i].vrollmask == BX_VGA_THIS s.linfo[i].vrollmask &&
		  newinfo[i].paloffset == BX_VGA_THIS s.linfo[i].paloffset) {
	  if(newinfo[i].displeft == BX_VGA_THIS s.linfo[i].displeft &&
	     newinfo[i].dispoffh == BX_VGA_THIS s.linfo[i].dispoffh &&
	     newinfo[i].dispoffl == BX_VGA_THIS s.linfo[i].dispoffl) {
	    /* don't need update */
	  } else {
	    /* partial update (not yet) */
	    BX_VGA_THIS s.all_updated = 1;
	    BX_VGA_THIS s.linfo[i]    = newinfo[i];
	  }
	} else {
	  /* update */
	  BX_VGA_THIS s.all_updated = 1;
	  BX_VGA_THIS s.linfo[i]    = newinfo[i];
	}
      }
    } else {
      BX_INFO(("vga : display stop\n",i));
      if(BX_VGA_THIS s.linfo[0].depth!=0 ||
	 BX_VGA_THIS s.linfo[1].depth!=0) {
	BX_VGA_THIS s.linfo[0].depth=BX_VGA_THIS s.linfo[1].depth=0;
	bx_gui.clear_screen();
      }
    }
  }
  BX_VGA_THIS s.dispmode_updated=0;

  if(BX_VGA_THIS s.linfo[0].depth==0 &&
     BX_VGA_THIS s.linfo[1].depth==0) return;

  if(BX_VGA_THIS s.partial_updated ||
     BX_VGA_THIS s.all_updated) {
    for (yti=0; yti<(BX_VGA_THIS s.y_dispsize+Y_TILESIZE-1)/Y_TILESIZE; yti++) {
      for (xti=0; xti<(BX_VGA_THIS s.x_dispsize+X_TILESIZE-1)/X_TILESIZE; xti++) {
	if (BX_VGA_THIS s.vga_tile_updated[xti][yti] ||
	    BX_VGA_THIS s.all_updated) {
	  for (y=0; y<Y_TILESIZE; y++) {
	    for (x=0; x<X_TILESIZE; x++) {
	      BX_VGA_THIS s.tile[y*X_TILESIZE + x] = 0;
	    }
	  }
	  for(i=0;i<2;i++) {
	    if(BX_VGA_THIS s.linfo[i].depth==0) break;
	    switch(BX_VGA_THIS s.linfo[i].depth) {
	    case 4:
	      for (y=0; y<Y_TILESIZE; y++) {
		if(y+yti*Y_TILESIZE<BX_VGA_THIS s.linfo[i].top ||
		   y+yti*Y_TILESIZE>BX_VGA_THIS s.linfo[i].bottom) continue;
		for (x=0; x<X_TILESIZE; x++) {
		  if(x+xti*X_TILESIZE>BX_VGA_THIS s.linfo[i].right) continue;
		  if(x+xti*X_TILESIZE>=BX_VGA_THIS s.linfo[i].left) {
		    addr = BX_VGA_THIS s.linfo[i].layeraddr+
		      ((BX_VGA_THIS s.linfo[i].dispoffh
			+((BX_VGA_THIS s.linfo[i].dispoffl
			   +(x+xti*X_TILESIZE-BX_VGA_THIS s.linfo[i].displeft)
			   *2/BX_VGA_THIS s.linfo[i].xzoom2/2)
			  & BX_VGA_THIS s.linfo[i].vrollmask)
			+((y+yti*Y_TILESIZE)-BX_VGA_THIS s.linfo[i].top)
			/BX_VGA_THIS s.linfo[i].yzoom
			*BX_VGA_THIS s.linfo[i].lineoff)
		       &BX_VGA_THIS s.linfo[i].layermask);
		    xparity=((x+xti*X_TILESIZE-BX_VGA_THIS s.linfo[i].displeft)
			     *2/BX_VGA_THIS s.linfo[i].xzoom2)&1;
		    //BX_INFO(("vga:%d %d %x\n",x+xti*X_TILESIZE,y+yti*Y_TILESIZE,addr));
		    if(xparity==0) {
		      pel=BX_VGA_THIS s.video_memory[addr] & 0x0f;
		    } else {
		      pel=(BX_VGA_THIS s.video_memory[addr] & 0xf0)>>4;
		    }
		    //BX_INFO(("vga:%d %d %d %d\n",i,x+xti*X_TILESIZE,y+yti*Y_TILESIZE,pel));
		    if(pel!=0) {
		      BX_VGA_THIS s.tile[y*X_TILESIZE + x] = pel |
			BX_VGA_THIS s.linfo[i].paloffset;
		    }
		  }
		}
	      }
	      break;

	    case 8:
	      for (y=0; y<Y_TILESIZE; y++) {
		if(y+yti*Y_TILESIZE<BX_VGA_THIS s.linfo[i].top ||
		   y+yti*Y_TILESIZE>BX_VGA_THIS s.linfo[i].bottom) continue;
		for (x=0; x<X_TILESIZE; x++) {
		  if(x+xti*X_TILESIZE>BX_VGA_THIS s.linfo[i].right) continue;
		  if(x+xti*X_TILESIZE>=BX_VGA_THIS s.linfo[i].left) {
		    addr = BX_VGA_THIS s.linfo[i].layeraddr+
		      ((BX_VGA_THIS s.linfo[i].dispoffh
			+((BX_VGA_THIS s.linfo[i].dispoffl
			   +(x+xti*X_TILESIZE-BX_VGA_THIS s.linfo[i].displeft)
			   *2/BX_VGA_THIS s.linfo[i].xzoom2)
			  & BX_VGA_THIS s.linfo[i].vrollmask)
			+((y+yti*Y_TILESIZE)-BX_VGA_THIS s.linfo[i].top)
			/BX_VGA_THIS s.linfo[i].yzoom
			*BX_VGA_THIS s.linfo[i].lineoff)
		       &BX_VGA_THIS s.linfo[i].layermask);
		    //BX_INFO(("vga:%d %d %x\n",x+xti*X_TILESIZE,y+yti*Y_TILESIZE,addr));
		    pel=BX_VGA_THIS s.video_memory[addr];
		    //BX_INFO(("vga:%d %d %d %d\n",i,x+xti*X_TILESIZE,y+yti*Y_TILESIZE,pel));
		    if(pel!=0) {
		      BX_VGA_THIS s.tile[y*X_TILESIZE + x] = pel |
			BX_VGA_THIS s.linfo[i].paloffset;
		    }
		  }
		}
	      }
	      break;

	    case 16:
	      for (y=0; y<Y_TILESIZE; y++) {
		if(y+yti*Y_TILESIZE<BX_VGA_THIS s.linfo[i].top ||
		   y+yti*Y_TILESIZE>BX_VGA_THIS s.linfo[i].bottom) continue;
		for (x=0; x<X_TILESIZE; x++) {
		  if(x+xti*X_TILESIZE>BX_VGA_THIS s.linfo[i].right) continue;
		  if(x+xti*X_TILESIZE>=BX_VGA_THIS s.linfo[i].left) {
		    addr = BX_VGA_THIS s.linfo[i].layeraddr+
		      ((BX_VGA_THIS s.linfo[i].dispoffh
			+((BX_VGA_THIS s.linfo[i].dispoffl
			   +(x+xti*X_TILESIZE-BX_VGA_THIS s.linfo[i].displeft)
			   *2/BX_VGA_THIS s.linfo[i].xzoom2*2)
			  & BX_VGA_THIS s.linfo[i].vrollmask)
			+((y+yti*Y_TILESIZE)-BX_VGA_THIS s.linfo[i].top)
			/BX_VGA_THIS s.linfo[i].yzoom
			*BX_VGA_THIS s.linfo[i].lineoff)
		       &BX_VGA_THIS s.linfo[i].layermask);
		    //BX_INFO(("vga:%d %d %x\n",x+xti*X_TILESIZE,y+yti*Y_TILESIZE,addr));
#if WORDS_BIGENDIAN
		    pel=(BX_VGA_THIS s.video_memory[addr] << 8) |
		      BX_VGA_THIS s.video_memory[addr + 1];
#else
		    pel=*((Bit16u*)(&BX_VGA_THIS s.video_memory[addr]));
#endif
		    //BX_INFO(("vga:%d %d %d %d\n",i,x+xti*X_TILESIZE,y+yti*Y_TILESIZE,pel));
		    if((pel & 0x8000)==0) {
		      BX_VGA_THIS s.tile[y*X_TILESIZE + x] = pel |
			BX_VGA_THIS s.linfo[i].paloffset;
		    }
		  }
		}
	      }
	      break;

	    default:
	      BX_INFO(("vga: unsupported depth %d.\n",BX_VGA_THIS s.linfo[i].depth));
	    }
	  }
	  bx_gui.graphics_tile_update(BX_VGA_THIS s.tile,
				      xti*X_TILESIZE, yti*Y_TILESIZE);
	  BX_VGA_THIS s.vga_tile_updated[xti][yti] = 0;
	}
      }
    }
    BX_VGA_THIS s.partial_updated=0;
    BX_VGA_THIS s.all_updated=0;
  }
}

#define DRAW16(AT,XS,YS,XP,YP) \
case AT: \
  y2=y; \
  for(yc=0;yc<YS;yc++) { \
    x2=x; \
    if(y2<256) { \
      for(xc=0;xc<XS;xc++) { \
	if(x2<256) { \
	  p16=ptop16[(XP)+(YP)*16]; \
	  if((p16 & 0x8000)==0) { \
	    vtop[x2+y2*256]=p16; \
	  } \
	} \
	x2=(x2+1)&0x1ff; \
      } \
    } \
    y2=(y2+1)&0x1ff; \
  } \
  break;

#define DRAW4N(AT,XS,YS,XP,YP) \
case AT: \
  y2=y; \
  for(yc=0;yc<YS;yc++) { \
    x2=x; \
    if(y2<256) { \
      for(xc=0;xc<XS;xc+=2) { \
	if(x2<256) { \
	  p4=ptop4[(XP)/2+(YP)*8]&0xf; \
	  if(p4!=0) { \
	    vtop[x2+y2*256]=ctop4[p4]; \
	  } \
	} \
	x2=(x2+1)&0x1ff; \
	if(x2<256) { \
	  p4=ptop4[(XP)/2+(YP)*8]>>4; \
	  if(p4!=0) { \
	    vtop[x2+y2*256]=ctop4[p4]; \
	  } \
	} \
	x2=(x2+1)&0x1ff; \
     } \
    } \
    y2=(y2+1)&0x1ff; \
  } \
  break;

  void
bx_vga_c::sprite_update(void)
{
  int max_sprite;
  Bit16u xoff,yoff;
  Bit16u *vtop;
  Bit16u *stop;
  Bit16u *ptop16;
  Bit16u *ctop4;
  Bit8u *ptop4;
  Bit16u x,y,attr,color;
  int i;
  int x2,y2,xc,yc;
  Bit16u p16;
  Bit8u p4;

  if((BX_VGA_THIS s.sprite.reg[1]&0x80)==0) return;

  max_sprite = BX_VGA_THIS s.sprite.reg[0] |
    ((BX_VGA_THIS s.sprite.reg[1] << 8) & 0x300);
  //if(max_sprite==0) {
  //max_sprite=1024;
  //} else {
  //max_sprite--;
  //}

  // printf("sssssss %d\n",max_sprite);

  xoff = BX_VGA_THIS s.sprite.reg[2] |
    (BX_VGA_THIS s.sprite.reg[3] << 8) & 0x100;
  yoff = BX_VGA_THIS s.sprite.reg[4] |
    (BX_VGA_THIS s.sprite.reg[5] << 8) & 0x100;

  BX_VGA_THIS s.sprite.reg[6] ^= 0x10;
  BX_VGA_THIS s.all_updated      = 1;
  BX_VGA_THIS s.dispmode_updated = 1;

  vtop = (Bit16u*)(&BX_VGA_THIS
		   s.video_memory[
				  (BX_VGA_THIS s.sprite.reg[6] & 0x10) == 0
				  ? 0x60000 : 0x40000]);
  stop = (Bit16u*)(BX_VGA_THIS s.sprite.mem);

  for(i=0;i<0x10000;i++) {
    vtop[i] = 0x8000;
  }

  //for(i=0;i<=max_sprite;i++) {
  for(i=max_sprite;i<1024;i++) {
    //    if(i<2) {
    // printf("%.4x %.4x %.4x %.4x %.4x\n",
    //	     i,
    //	     stop[i*4+0],
    //	     stop[i*4+1],
    //	     stop[i*4+2],
    //	     stop[i*4+3]);
    //}
    color  = stop[i*4+3];
    if((color & 0x2000) == 0) {
      x    = stop[i*4+0];
      y    = stop[i*4+1];
      attr = stop[i*4+2];
      //printf("%.4x %.4x %.4x %.4x %.4x\n",i,x,y,attr,color);
      if((attr & 0x8000) != 0) {
	x += xoff;
	y += yoff;
      }
      x &= 0x1ff;
      y &= 0x1ff;
      switch(color & 0xc000) {
      case 0x0000:
	ptop16 = (Bit16u*)(&BX_VGA_THIS s.sprite.mem[128*(attr & 0x3ff)]);
	switch(attr & 0x7c00) {
	  DRAW16(0x0000,16,16,xc,yc);
	default:
	  printf("%.4x %.4x\n",attr,color);
	}
	break;
      case 0x8000:
	ptop4 = &BX_VGA_THIS s.sprite.mem[128*(attr & 0x3ff)];
	ctop4 = (Bit16u*)(&BX_VGA_THIS s.sprite.mem[32*(color & 0xfff)]);
	switch(attr & 0x7c00) {
	  DRAW4N(0x0000,16,16,xc,yc);
	default:
	  printf("%.4x %.4x\n",attr,color);
	}
	break;
      default:
	printf("%.4x %.4x\n",attr,color);
      }
    }
  }
}


  Bit8u
bx_vga_c::mem_read(Bit32u addr)
{
  Bit32u baseaddr;

  //if (bx_dbg.video)
  //BX_INFO(("vga : read from video memory 0x%.8x\n",addr));

  if(addr>=0xc0000 && addr<=0xc7fff) {
    baseaddr=((addr&0x7fff) << 2) |
      (BX_VGA_THIS s.FMR_compat.access_page_select ? 0x20000 : 0x00000);
    return((((BX_VGA_THIS s.video_memory[baseaddr]
	      >> BX_VGA_THIS s.FMR_compat.write_plane) << 7) & 0x80) |
	   (((BX_VGA_THIS s.video_memory[baseaddr]
	      >> BX_VGA_THIS s.FMR_compat.write_plane) << 2) & 0x40) |
	   (((BX_VGA_THIS s.video_memory[baseaddr + 1]
	      >> BX_VGA_THIS s.FMR_compat.write_plane) << 5) & 0x20) |
	   ( (BX_VGA_THIS s.video_memory[baseaddr + 1]
	      >> BX_VGA_THIS s.FMR_compat.write_plane)       & 0x10) |
	   (((BX_VGA_THIS s.video_memory[baseaddr + 2]
	      >> BX_VGA_THIS s.FMR_compat.write_plane) << 3) & 0x08) |
	   (((BX_VGA_THIS s.video_memory[baseaddr + 2]
	      >> BX_VGA_THIS s.FMR_compat.write_plane) >> 2) & 0x04) |
	   (((BX_VGA_THIS s.video_memory[baseaddr + 3]
	      >> BX_VGA_THIS s.FMR_compat.write_plane) << 1) & 0x02) |
	   (((BX_VGA_THIS s.video_memory[baseaddr + 3]
	      >> BX_VGA_THIS s.FMR_compat.write_plane) >> 4) & 0x01));
  } else if((addr>=0xc8000 && addr<=0xc8fff) ||
	    (addr>=0xca000 && addr<=0xcafff)) {
    return(BX_VGA_THIS s.sprite.mem[addr-0xc8000]);
  } else if(addr>=0xcff80 && addr<=0xcffdf) {
    // memory maped I/O
    switch(addr&0xffff) {
    case 0xff80:
      return(BX_VGA_THIS s.FMR_compat.MIX_reg);

    case 0xff81:
      return( (BX_VGA_THIS s.FMR_compat.write_plane      & 0x0f) |
	     ((BX_VGA_THIS s.FMR_compat.read_plane << 6) & 0xc0));

    case 0xff96:
      //BX_INFO(("vga: %d\n",BX_VGA_THIS s.FMR_compat.kanji_row_count));
      //return(BX_VGA_THIS s.FMR_compat.kanji_row_count);
      return(BX_VGA_THIS devices->mem->fntrom[(BX_VGA_THIS s.FMR_compat.kanji_offset |
					       ((BX_VGA_THIS s.FMR_compat.kanji_row_count&0xf)<<1)
					       )&0x3ffff]);

    case 0xff97:
      return(BX_VGA_THIS devices->mem->fntrom[(BX_VGA_THIS s.FMR_compat.kanji_offset |
					       (((BX_VGA_THIS s.FMR_compat.kanji_row_count++)&0xf)<<1) |
					       1
					       )&0x3ffff]);

    default:
      BX_PANIC(("vga : read from unknown memory maped I/O 0xc000:0x%.4x\n",addr&0xffff));
      return(0xff);
    }
  } else if((addr>=0x80000000 && addr<0x80080000) ||
            (addr>=0x80100000 && addr<0x80180000)) {
    // VRAM
    return(BX_VGA_THIS s.video_memory[addr&0x0007ffff]);
  } else if(addr>=0x81000000 && addr<0x81020000) {
    // sprite RAM
    return(BX_VGA_THIS s.sprite.mem[addr&0x0001ffff]);
  } else {
    BX_PANIC(("vga : read from unknown video memory 0x%.8x\n",addr));
  }
  return(0);
}

void
bx_vga_c::mem_update(Bit32u addr, Bit8u value)
{
  int x,y,xo,yo;
  int i;

  if(BX_VGA_THIS s.video_memory[addr] != value) {
    //BX_VGA_THIS s.all_updated = 1;
    //BX_INFO(("vga : %x %x %d %d\n",addr,value,((addr&0x3ffff)%512)*2,(addr&0x3ffff)/512));
    BX_VGA_THIS s.video_memory[addr]=value;
    for(i=0;i<2;i++) {
      if(BX_VGA_THIS s.linfo[i].depth!=0) {
	if((addr&(~BX_VGA_THIS s.linfo[i].layermask))
	   ==BX_VGA_THIS s.linfo[i].layeraddr) {
	  y=((addr
	      -BX_VGA_THIS s.linfo[i].layeraddr
	      -BX_VGA_THIS s.linfo[i].dispoffh)
	     &BX_VGA_THIS s.linfo[i].layermask)
	    /BX_VGA_THIS s.linfo[i].lineoff
	    *BX_VGA_THIS s.linfo[i].yzoom
	    +BX_VGA_THIS s.linfo[i].top;
	  x=((addr
	      -BX_VGA_THIS s.linfo[i].layeraddr
	      -BX_VGA_THIS s.linfo[i].dispoffh
	      -(y-BX_VGA_THIS s.linfo[i].top)
	      /BX_VGA_THIS s.linfo[i].yzoom*
	      BX_VGA_THIS s.linfo[i].lineoff
	      -BX_VGA_THIS s.linfo[i].dispoffl)
	     &BX_VGA_THIS s.linfo[i].vrollmask)*8
	    /BX_VGA_THIS s.linfo[i].depth
	    *BX_VGA_THIS s.linfo[i].xzoom2/2
	    +BX_VGA_THIS s.linfo[i].displeft;
	  for(yo=y;yo<y+BX_VGA_THIS s.linfo[i].yzoom;yo++) {
	    for(xo=x;xo<x+(BX_VGA_THIS s.linfo[i].xzoom2+1)/2
		  *(BX_VGA_THIS s.linfo[i].depth==4 ? 2 :1);xo++) {
#if 0
	      if (bx_dbg.video)
		BX_INFO(("vga : update (%d %d %d)\n",i,xo,yo));
#endif
	      if(yo<BX_VGA_THIS s.linfo[i].top ||
		 yo>BX_VGA_THIS s.linfo[i].bottom ||
		 xo<BX_VGA_THIS s.linfo[i].left ||
		 xo>BX_VGA_THIS s.linfo[i].right) continue;
	      BX_VGA_THIS s.partial_updated = 1;
	      BX_VGA_THIS s.vga_tile_updated[xo/X_TILESIZE][yo/Y_TILESIZE] = 1;
	    }
	  }
	}
      }
    }
  }
}

  void
bx_vga_c::mem_write(Bit32u addr, Bit8u value)
{
  Bit32u baseaddr;

  //if (bx_dbg.video)
  //BX_INFO(("vga : write 0x%.2x to video memory 0x%.8x\n",value,addr));

  if(addr>=0xc0000 && addr<=0xc7fff) {
    baseaddr=((addr&0x7fff) << 2) |
      (BX_VGA_THIS s.FMR_compat.access_page_select ? 0x20000 : 0x00000);
    mem_update(baseaddr,
	       (BX_VGA_THIS s.video_memory[baseaddr] &
		((BX_VGA_THIS s.FMR_compat.write_plane*0x11)^0xff)) |
	       ((((value & 0x80) >> 7) |
		 ((value & 0x40) >> 2)) *
		BX_VGA_THIS s.FMR_compat.write_plane));
    mem_update(baseaddr+1,
	       (BX_VGA_THIS s.video_memory[baseaddr+1] &
		((BX_VGA_THIS s.FMR_compat.write_plane*0x11)^0xff)) |
	       ((((value & 0x20) >> 5) |
		 ((value & 0x10)     )) *
		BX_VGA_THIS s.FMR_compat.write_plane));
    mem_update(baseaddr+2,
	       (BX_VGA_THIS s.video_memory[baseaddr+2] &
		((BX_VGA_THIS s.FMR_compat.write_plane*0x11)^0xff)) |
	       ((((value & 0x08) >> 3) |
		 ((value & 0x04) << 2)) *
		BX_VGA_THIS s.FMR_compat.write_plane));
    mem_update(baseaddr+3,
	       (BX_VGA_THIS s.video_memory[baseaddr+3] &
		((BX_VGA_THIS s.FMR_compat.write_plane*0x11)^0xff)) |
	       ((((value & 0x02) >> 1) |
		 ((value & 0x01) << 4)) *
		BX_VGA_THIS s.FMR_compat.write_plane));
  } else if((addr>=0xc8000 && addr<=0xc8fff) ||
	    (addr>=0xca000 && addr<=0xcafff)) {
    BX_VGA_THIS s.FMR_compat.TVRAM_MD=1;
    BX_VGA_THIS s.sprite.mem[addr-0xc8000]=value;
    return;
  } else if(addr>=0xcff80 && addr<=0xcffdf) {
    // memory maped I/O
    switch(addr&0xffff) {
    case 0xff80:
      BX_VGA_THIS s.FMR_compat.MIX_reg = value;
      return;

    case 0xff81:
      BX_VGA_THIS s.FMR_compat.write_plane        =  (value&0x0f);
      BX_VGA_THIS s.FMR_compat.read_plane         = ((value&0xc0) >> 6);
      return;

    case 0xff82:
      BX_VGA_THIS s.FMR_compat.disp_plane         =  (value&0x07) |
                                                    ((value&0x20) >> 2);
      BX_VGA_THIS s.FMR_compat.disp_page_select   = ((value&0x10) != 0);
      return;

    case 0xff83:
      BX_VGA_THIS s.FMR_compat.access_page_select = ((value&0x10) != 0);
      return;

    case 0xff8e:
    case 0xff8f:
      BX_INFO(("vga : write 0x%.2x to unknown memory maped I/O 0xc000:0x%.4x(FM-R CRTC?, ignored)\n",value,addr&0xffff));
      return;

    case 0xff94:
      BX_VGA_THIS s.FMR_compat.kanji_code_H = value&0x7f;
      return;

    case 0xff95:
      BX_VGA_THIS s.FMR_compat.kanji_code_L = value&0x7f;
      BX_VGA_THIS s.FMR_compat.kanji_row_count = 0;
      if(BX_VGA_THIS s.FMR_compat.kanji_code_H<0x30) {
	BX_VGA_THIS s.FMR_compat.kanji_offset =
	  (((BX_VGA_THIS s.FMR_compat.kanji_code_L - 0x00) & 0x1f) <<  5) |
	  (((BX_VGA_THIS s.FMR_compat.kanji_code_L - 0x20) & 0x20) <<  9) |
	  (((BX_VGA_THIS s.FMR_compat.kanji_code_L - 0x20) & 0x40) <<  7) |
	  (((BX_VGA_THIS s.FMR_compat.kanji_code_H - 0x00) & 0x07) << 10) |
	  0x00000;
      } else if(BX_VGA_THIS s.FMR_compat.kanji_code_H<0x70) {
	BX_VGA_THIS s.FMR_compat.kanji_offset =
	  (((BX_VGA_THIS s.FMR_compat.kanji_code_L - 0x00) & 0x1f) <<  5) +
	  (((BX_VGA_THIS s.FMR_compat.kanji_code_L - 0x20) & 0x60) <<  9) +
	  (((BX_VGA_THIS s.FMR_compat.kanji_code_H - 0x00) & 0x0f) << 10) +
	  (((BX_VGA_THIS s.FMR_compat.kanji_code_H - 0x30) & 0x70) * 0xc00) +
	  0x08000;
      } else {
	BX_VGA_THIS s.FMR_compat.kanji_offset =
	  (((BX_VGA_THIS s.FMR_compat.kanji_code_L - 0x00) & 0x1f) <<  5) |
	  (((BX_VGA_THIS s.FMR_compat.kanji_code_L - 0x20) & 0x20) <<  9) |
	  (((BX_VGA_THIS s.FMR_compat.kanji_code_L - 0x20) & 0x40) <<  7) |
	  (((BX_VGA_THIS s.FMR_compat.kanji_code_H - 0x00) & 0x07) << 10) |
	  0x38000;
      }
      return;

    case 0xff99:
      BX_VGA_THIS devices->mem->ANKCG = ((value & 0x01)!=0);
      return;

    default:
      BX_PANIC(("vga : write 0x%.2x to unknown memory maped I/O 0xc000:0x%.4x\n",value,addr&0xffff));
      return;
    }
  } else if((addr>=0x80000000 && addr<0x80080000) ||
	    (addr>=0x80100000 && addr<0x80180000)) {
    // VRAM
    mem_update(addr&0x0007ffff,
	       (BX_VGA_THIS s.video_memory[addr&0x0007ffff]
		&(~BX_VGA_THIS s.vram_ctrl.mask[addr&3]))|
	       (value&(BX_VGA_THIS s.vram_ctrl.mask[addr&3])));
    return;

    BX_VGA_THIS s.sprite.mem[addr-0xc8000]=value;
  } else if(addr>=0x81000000 && addr<0x81020000) {
    // sprite RAM
    //printf("S %.5x => %.4x\n",addr&0x0001ffff,value);
    BX_VGA_THIS s.sprite.mem[addr&0x0001ffff]=value;
    return;
  } else {
    BX_PANIC(("vga : write 0x%.2x to unknown video memory 0x%.8x\n",value,addr));
  }
}



  void
bx_vga_c::dump_status(void)
{
}


  void
bx_vga_c::redraw_area(unsigned x0, unsigned y0, unsigned width,
                      unsigned height)
{
  unsigned xi, yi, x1, y1;

  BX_VGA_THIS s.partial_updated = 1;

  x1 = x0 + width  - 1;
  y1 = y0 + height - 1;

  for (yi=0; yi<BX_VGA_THIS s.y_dispsize; yi+=Y_TILESIZE) {
    for (xi=0; xi<BX_VGA_THIS s.x_dispsize; xi+=X_TILESIZE) {
      // is redraw rectangle outside x boundaries of this tile?
      if (x1 < xi) continue;
      if (x0 > (xi+X_TILESIZE-1)) continue;

      // is redraw rectangle outside y boundaries of this tile?
      if (y1 < yi) continue;
      if (y0 > (yi+X_TILESIZE-1)) continue;
      BX_VGA_THIS s.vga_tile_updated[xi/X_TILESIZE][yi/Y_TILESIZE] = 1;
    }
  }
}


#ifdef VIDEO_MODE_TEST
typedef enum {
  NONE,
  CYLINDRICAL,
  SPHRICAL} scrolltype;

typedef struct {
  int regsetno;
  Bit16u crtc[32];
  Bit8u shifter[2];
  int numlayers;
  int modes[2];
} videomode;

typedef struct {
  int vxsize,vysize;
  int dxsize,dysize;
  int depth;
  scrolltype st;
  Boolean sprite;
} modeinfo;

modeinfo modes[]={
  {},
  { 640,400,640,400, 4,       NONE,0},
  { 640,400,640,200, 4,       NONE,0},
  {1024,512,640,480, 4,CYLINDRICAL,0},
  {1024,512,640,400, 4,CYLINDRICAL,0},
  { 256,512,256,256,16,CYLINDRICAL,1},
  { 256,512,256,256,16,CYLINDRICAL,1},
  { 256,512,256,240,16,CYLINDRICAL,1},
  { 256,512,256,240,16,CYLINDRICAL,1},
  { 512,256,360,240,16,   SPHRICAL,0},
  { 512,256,320,240,16,   SPHRICAL,0},
  { 512,256,320,240,16,   SPHRICAL,0},
  {1024,512,640,480, 8,CYLINDRICAL,0},
  {1024,512,640,400, 8,CYLINDRICAL,0},
  {1024,512,720,480, 8,CYLINDRICAL,0},
  { 512,512,320,480,16,CYLINDRICAL,0},
  { 512,512,320,480,16,CYLINDRICAL,0},
  { 512,512,512,480,16,CYLINDRICAL,0},
  { 512,512,512,480,16,CYLINDRICAL,0}
};

videomode regsets[] = {
  {1,
   {0x0060,0x02c0,0x0000,0x0000,0x031f,0x0000,0x0004,0x0000,
    0x0419,0x008a,0x030a,0x008a,0x030a,0x0046,0x0406,0x0046,
    0x0406,0x0000,0x008a,0x0000,0x0080,0x0000,0x008a,0x0000,
    0x0080,0x0058,0x0001,0x0000,0x000f,0x0002,0x0000,0x0192},
   {0x0a,0x18},
   1,{12}},
  {2,
   {0x0040,0x0320,0x0000,0x0000,0x035f,0x0000,0x0010,0x0000,
    0x036f,0x009c,0x031c,0x009c,0x031c,0x0040,0x0360,0x0040,
    0x0360,0x0000,0x009c,0x0000,0x0080,0x0000,0x009c,0x0000,
    0x0080,0x004a,0x0001,0x0000,0x000f,0x0003,0x0000,0x0150},
   {0x0a,0x18},
   1,{13}},
  {3,
   {0x0086,0x0610,0x0000,0x0000,0x071b,0x0006,0x000c,0x0012,
    0x020c,0x0129,0x06c9,0x0129,0x06c9,0x002a,0x020a,0x002a,
    0x020a,0x0000,0x0129,0x0080,0x0100,0x0000,0x0129,0x0080,
    0x0100,0x0064,0x0007,0x0101,0x000f,0x000c,0x0003,0x01ca},
   {0x0a,0x18},
   1,{14}},
  {4,
   {0x0060,0x02c0,0x0000,0x0000,0x031f,0x0000,0x0004,0x0000,
    0x0419,0x008a,0x030a,0x008a,0x030a,0x0046,0x0406,0x0046,
    0x0406,0x0000,0x008a,0x0000,0x0080,0x0000,0x008a,0x0000,
    0x0080,0x0058,0x0001,0x0101,0x000a,0x0002,0x0000,0x0192},
   {0x0f,0x08},
   1,{15}},
  {5,
   {0x0074,0x0530,0x0000,0x0000,0x0617,0x0006,0x000c,0x0012,
    0x020c,0x00e7,0x05e7,0x00e7,0x05e7,0x002a,0x020a,0x002a,
    0x020a,0x0000,0x00e7,0x0080,0x0100,0x0000,0x00e7,0x0080,
    0x0100,0x0056,0x0007,0x0303,0x000a,0x0001,0x0002,0x0188},
   {0x0f,0x08},
   1,{16}},
  {6,
   {0x0040,0x0320,0x0000,0x0000,0x035f,0x0000,0x0010,0x0000,
    0x036f,0x009c,0x031c,0x009c,0x031c,0x0040,0x0360,0x0040,
    0x0360,0x0000,0x009c,0x0000,0x0050,0x0000,0x009c,0x0000,
    0x0050,0x004a,0x0001,0x0000,0x003f,0x0003,0x0000,0x0150},
   {0x15,0x08},
   2,{1,1}},
  {8,
   {0x0040,0x0320,0x0000,0x0000,0x035f,0x0000,0x0010,0x0000,
    0x036f,0x009c,0x031c,0x009c,0x031c,0x0040,0x0360,0x0040,
    0x0360,0x0000,0x009c,0x0000,0x0050,0x0000,0x009c,0x0000,
    0x0050,0x004a,0x0001,0x1010,0x003f,0x0003,0x0000,0x0150},
   {0x15,0x08},
   2,{2,2}},
  {9,
   {0x0060,0x02c0,0x0000,0x0000,0x031f,0x0000,0x0004,0x0000,
    0x0419,0x008a,0x030a,0x008a,0x030a,0x0046,0x0406,0x0046,
    0x0406,0x0000,0x008a,0x0000,0x0080,0x0000,0x008a,0x0000,
    0x0080,0x0058,0x0001,0x0000,0x000f,0x0002,0x0000,0x0192},
   {0x15,0x08},
   2,{3,3}},
  {10,
   {0x0040,0x0320,0x0000,0x0000,0x035f,0x0000,0x0010,0x0000,
    0x036f,0x009c,0x031c,0x009c,0x031c,0x0040,0x0360,0x0040,
    0x0360,0x0000,0x009c,0x0000,0x0080,0x0000,0x009c,0x0000,
    0x0080,0x004a,0x0001,0x0000,0x000f,0x0003,0x0000,0x0150},
   {0x15,0x08},
   2,{4,4}},
  {11,
   {0x0040,0x0320,0x0000,0x0000,0x035f,0x0000,0x0010,0x0000,
    0x036f,0x009c,0x019c,0x009c,0x031c,0x0040,0x0240,0x0040,
    0x0360,0x0000,0x009c,0x0000,0x0080,0x0000,0x009c,0x0000,
    0x0080,0x004a,0x0001,0x0000,0x000d,0x0003,0x0000,0x0150},
   {0x17,0x08},
   2,{6,4}},
  {12,
   {0x0086,0x0610,0x0000,0x0000,0x071b,0x0006,0x000c,0x0012,
    0x020c,0x0129,0x06c9,0x0129,0x06c9,0x002a,0x020a,0x002a,
    0x020a,0x0000,0x0129,0x0000,0x0100,0x0000,0x0129,0x0000,
    0x0100,0x0064,0x0007,0x0303,0x0005,0x000c,0x0003,0x01ca},
   {0x1f,0x08},
   2,{9,9}},
  {13,
   {0x0086,0x0610,0x0000,0x0000,0x071b,0x0006,0x000c,0x0012,
    0x020c,0x0129,0x0529,0x0129,0x06c9,0x002a,0x020a,0x002a,
    0x020a,0x0000,0x0129,0x0000,0x0080,0x0000,0x0129,0x0000,
    0x0100,0x0064,0x0007,0x0303,0x0005,0x000c,0x0003,0x01ca},
   {0x1f,0x08},
   2,{7,9}},
  {14,
   {0x0074,0x0530,0x0000,0x0000,0x0617,0x0006,0x000c,0x0012,
    0x020b,0x00e7,0x05e7,0x00e7,0x05e7,0x002a,0x020a,0x002a,
    0x020a,0x0000,0x00e7,0x0000,0x0100,0x0000,0x00e7,0x0000,
    0x0100,0x0056,0x0007,0x0303,0x0005,0x0001,0x0002,0x0188},
   {0x1f,0x08},
   2,{11,11}},
  {15,
   {0x0074,0x0530,0x0000,0x0000,0x0617,0x0006,0x000c,0x0012,
    0x020b,0x00e7,0x04e7,0x00e7,0x05e7,0x002a,0x020a,0x002a,
    0x020a,0x0000,0x00e7,0x0000,0x0080,0x0000,0x00e7,0x0000,
    0x0100,0x0056,0x0007,0x0303,0x0005,0x0001,0x0002,0x0188},
   {0x1f,0x08},
   2,{8,11}},
  {16,
   {0x0040,0x0320,0x0000,0x0000,0x035f,0x0000,0x0010,0x0000,
    0x036f,0x009c,0x019c,0x009c,0x019c,0x0040,0x0240,0x0040,
    0x0240,0x0000,0x009c,0x0000,0x0080,0x0000,0x009c,0x0000,
    0x0080,0x004a,0x0001,0x0000,0x0005,0x0003,0x0000,0x0150},
   {0x1f,0x08},
   2,{6,6}},
  {17,
   {0x0086,0x0610,0x0000,0x0000,0x071b,0x0006,0x000c,0x0012,
    0x020c,0x0129,0x0529,0x0129,0x0529,0x002a,0x020a,0x002a,
    0x020a,0x0000,0x0129,0x0000,0x0080,0x0000,0x0129,0x0000,
    0x0080,0x0064,0x0007,0x0303,0x0005,0x000c,0x0003,0x01ca},
   {0x1f,0x08},
   2,{7,7}},
  {18,
   {0x0074,0x0530,0x0000,0x0000,0x0617,0x0006,0x000c,0x0012,
    0x020b,0x00e7,0x04e7,0x00e7,0x04e7,0x002a,0x020a,0x002a,
    0x020a,0x0000,0x00e7,0x0000,0x0080,0x0000,0x00e7,0x0000,
    0x0080,0x0056,0x0007,0x0303,0x0005,0x0001,0x0002,0x0188},
   {0x1f,0x08},
   2,{8,8}},
  {19,
   {0x0060,0x02c0,0x0000,0x0000,0x031f,0x0000,0x0004,0x0000,
    0x0419,0x008a,0x018a,0x008a,0x030a,0x0046,0x0246,0x0046,
    0x0406,0x0000,0x008a,0x0000,0x0080,0x0000,0x008a,0x0000,
    0x0080,0x0058,0x0001,0x0000,0x000d,0x0002,0x0000,0x0192},
   {0x17,0x08},
   2,{5,3}},
  {20,
   {0x0060,0x02c0,0x0000,0x0000,0x031f,0x0000,0x0004,0x0000,
    0x0419,0x008a,0x018a,0x008a,0x018a,0x0046,0x0246,0x0046,
    0x0246,0x0000,0x008a,0x0000,0x0080,0x0000,0x008a,0x0000,
    0x0080,0x0058,0x0001,0x0000,0x0005,0x0002,0x0000,0x0192},
   {0x1f,0x08},
   2,{5,5}},
  {22,
   {0x0060,0x02c0,0x0000,0x0000,0x031f,0x0000,0x0004,0x0000,
    0x0419,0x008a,0x030a,0x008a,0x018a,0x0046,0x0406,0x0046,
    0x0246,0x0000,0x008a,0x0000,0x0080,0x0000,0x008a,0x0000,
    0x0080,0x0058,0x0001,0x0000,0x0007,0x0002,0x0000,0x0192},
   {0x1d,0x08},
   2,{3,5}},
  {23,
   {0x0040,0x0320,0x0000,0x0000,0x035f,0x0000,0x0010,0x0000,
    0x036f,0x009c,0x031c,0x009c,0x019c,0x0040,0x0360,0x0040,
    0x0240,0x0000,0x009c,0x0000,0x0080,0x0000,0x009c,0x0000,
    0x0080,0x004a,0x0001,0x0000,0x0007,0x0003,0x0000,0x0150},
   {0x1d,0x08},
   2,{4,6}},
  {24,
   {0x0086,0x0610,0x0000,0x0000,0x071b,0x0006,0x000c,0x0012,
    0x020c,0x0129,0x06c9,0x0129,0x0529,0x002a,0x020a,0x002a,
    0x020a,0x0000,0x0129,0x0000,0x0100,0x0000,0x0129,0x0000,
    0x0080,0x0064,0x0007,0x0303,0x0005,0x000c,0x0003,0x01ca},
   {0x1f,0x08},
   2,{9,7}},
  {25,
   {0x0074,0x0530,0x0000,0x0000,0x0617,0x0006,0x000c,0x0012,
    0x020b,0x00e7,0x05e7,0x00e7,0x04e7,0x002a,0x020a,0x002a,
    0x020a,0x0000,0x00e7,0x0000,0x0100,0x0000,0x00e7,0x0000,
    0x0080,0x0056,0x0007,0x0303,0x0005,0x0001,0x0002,0x0188},
   {0x1f,0x08},
   2,{11,8}},
  {26,
   {0x0060,0x02c0,0x0000,0x0000,0x031f,0x0000,0x0004,0x0000,
    0x0419,0x008a,0x030a,0x008a,0x01ca,0x0046,0x0406,0x0046,
    0x0226,0x0000,0x008a,0x0000,0x0080,0x0000,0x008a,0x0000,
    0x0100,0x0058,0x0000,0x0000,0x0007,0x0002,0x0000,0x0192},
   {0x1d,0x08},
   2,{3,10}},
  {27,
   {0x0060,0x02c0,0x0000,0x0000,0x031f,0x0000,0x0004,0x0000,
    0x0419,0x008a,0x01ca,0x008a,0x030a,0x0046,0x0226,0x0046,
    0x0406,0x0000,0x008a,0x0000,0x0100,0x0000,0x008a,0x0000,
    0x0080,0x0058,0x0000,0x0000,0x000d,0x0002,0x0000,0x0192},
   {0x17,0x08},
   2,{10,3}},
  {28,
   {0x0060,0x02c0,0x0000,0x0000,0x031f,0x0000,0x0004,0x0000,
    0x0419,0x008a,0x01ca,0x008a,0x01ca,0x0046,0x0226,0x0046,
    0x0226,0x0000,0x008a,0x0000,0x0100,0x0000,0x008a,0x0000,
    0x0100,0x0058,0x0000,0x0000,0x0005,0x0002,0x0000,0x0192},
   {0x1f,0x08},
   2,{10,10}},
  {29,
   {0x0060,0x02c0,0x0000,0x0000,0x031f,0x0000,0x0004,0x0000,
    0x0419,0x008a,0x01ca,0x008a,0x018a,0x0046,0x0226,0x0046,
    0x0246,0x0000,0x008a,0x0000,0x0100,0x0000,0x008a,0x0000,
    0x0080,0x0058,0x0000,0x0000,0x0005,0x0002,0x0000,0x0192},
   {0x1f,0x08},
   2,{10,5}},
  {30,
#if 0
  /* incorrectly ? */
   {0x0060,0x02c0,0x0000,0x0000,0x031f,0x0000,0x0004,0x0000,
    0x0419,0x008a,0x01ca,0x008a,0x018a,0x0046,0x01e6,0x0046,
    0x0226,0x0000,0x008a,0x0000,0x0080,0x0000,0x008a,0x0000,
    0x0100,0x0058,0x0000,0x0000,0x0005,0x0002,0x0000,0x0192},
   {0x1f,0x08},
#else
   {0x0060,0x02c0,0x0000,0x0000,0x031f,0x0000,0x0004,0x0000,
    0x0419,0x008a,0x018a,0x008a,0x01ca,0x0046,0x0246,0x0046,
    0x0226,0x0000,0x008a,0x0000,0x0080,0x0000,0x008a,0x0000,
    0x0100,0x0058,0x0000,0x0000,0x0005,0x0002,0x0000,0x0192},
   {0x1f,0x08},
#endif
   2,{5,10}},
  {31,
   {0x0060,0x02c0,0x0000,0x0000,0x031f,0x0000,0x0004,0x0000,
    0x0419,0x00ca,0x02ca,0x00ca,0x02ca,0x0046,0x0406,0x0046,
    0x0406,0x0000,0x00ca,0x0000,0x0080,0x0000,0x00ca,0x0000,
    0x0080,0x0058,0x0001,0x0000,0x000a,0x0002,0x0000,0x0192},
   {0x0f,0x08},
   1,{17}},
  {32,
   {0x0074,0x0530,0x0000,0x0000,0x0617,0x0006,0x000c,0x0012,
    0x020c,0x0167,0x0567,0x0167,0x0567,0x002a,0x020a,0x002a,
    0x020a,0x0000,0x0167,0x0080,0x0100,0x0000,0x0167,0x0080,
    0x0100,0x0056,0x0001,0x0101,0x000a,0x0001,0x0002,0x0188},
   {0x0f,0x08},
   1,{18}},
  {-1}
};

struct {
  Bit8u red;
  Bit8u green;
  Bit8u blue;
} initial_pal16[16] = {
  {0x00,0x00,0x00},
  {0x00,0x00,0xc0},
  {0xc0,0x00,0x00},
  {0xc0,0x00,0xc0},
  {0x00,0xc0,0x00},
  {0x00,0xc0,0xc0},
  {0xc0,0xc0,0x00},
  {0xc0,0xc0,0xc0},
  {0x80,0x80,0x80},
  {0x00,0x00,0xf0},
  {0xf0,0x00,0x00},
  {0xf0,0x00,0xf0},
  {0x00,0xf0,0x00},
  {0x00,0xf0,0xf0},
  {0xf0,0xf0,0x00},
  {0xf0,0xf0,0xf0}};

#define UP() \
    BX_VGA_THIS update();\
    bx_gui.flush();\
    usleep(100000);\
    BX_VGA_THIS update();\
    bx_gui.flush();\

#define W() sleep(2);

  void
bx_vga_c::video_mode_test(void)
{
  int i,j,k;
  Bit32u addr;
  struct timeval start,end;

  i=(sizeof(regsets)/sizeof(videomode))-2;
  i=0;
  while(regsets[i].regsetno != -1) {
    regsets[i].crtc[28] |= 0x8000;
    regsets[i].shifter[1] &= 0xce;

    if(regsets[i].numlayers==1) {
      printf("register set no = %d, mode no = %d.\n",
	     regsets[i].regsetno,
	     regsets[i].modes[0]);
    } else {
      printf("register set no = %d, mode no = %d + %d.\n",
	     regsets[i].regsetno,
	     regsets[i].modes[0],
	     regsets[i].modes[1]);
    }
    // set video mode;
    for(j=0;j<32;j++) {
      if(j==2 || j==3) continue;
      BX_VGA_THIS write(0x0440,j,1,0);
      BX_VGA_THIS write(0x0442,regsets[i].crtc[j],2,0);
    }
    BX_VGA_THIS write(0x0448,0,1,0);
    BX_VGA_THIS write(0x044a,regsets[i].shifter[0],1,0);
    BX_VGA_THIS write(0x0448,1,1,0);
    BX_VGA_THIS write(0x044a,regsets[i].shifter[1],1,0);

    // draw test graphics and palette
    for(j=0;j<regsets[i].numlayers;j++) {
      switch(modes[regsets[i].modes[j]].depth) {
      case 4:
	for(addr=j*0x40000;
	    addr<j*0x40000+(0x80000/regsets[i].numlayers);
	    addr+=8) {
	  BX_VGA_THIS s.video_memory[addr  ] = 0x10;
	  BX_VGA_THIS s.video_memory[addr+1] = 0x32;
	  BX_VGA_THIS s.video_memory[addr+2] = 0x54;
	  BX_VGA_THIS s.video_memory[addr+3] = 0x76;
	  BX_VGA_THIS s.video_memory[addr+4] = 0x98;
	  BX_VGA_THIS s.video_memory[addr+5] = 0xba;
	  BX_VGA_THIS s.video_memory[addr+6] = 0xdc;
	  BX_VGA_THIS s.video_memory[addr+7] = 0xfd;
	}
	BX_VGA_THIS write(0x0448,1,1,0);
	BX_VGA_THIS write(0x044a,regsets[i].shifter[1] | (j * 0x20),1,0);
	for(k=0;k<16;k++) {
	  BX_VGA_THIS write(0xfd90,k,1,0);
	  BX_VGA_THIS write(0xfd92,initial_pal16[k].blue >> j,1,0);
	  BX_VGA_THIS write(0xfd94,initial_pal16[k].red >> j,1,0);
	  BX_VGA_THIS write(0xfd96,initial_pal16[k].green >> j,1,0);
	}
	break;

      case 8:
	for(addr=j*0x40000;
	    addr<j*0x40000+(0x80000/regsets[i].numlayers);
	    addr++) {
	  BX_VGA_THIS s.video_memory[addr] = addr;
	}
	BX_VGA_THIS write(0x0448,1,1,0);
	BX_VGA_THIS write(0x044a,regsets[i].shifter[1] | 0x10,1,0);
	for(k=0;k<256;k++) {
	  BX_VGA_THIS write(0xfd90,k,1,0);
	  BX_VGA_THIS write(0xfd92, ( k    &3)*0x55    ,1,0);
	  BX_VGA_THIS write(0xfd94,(((k>>2)&7)*0x49)>>1,1,0);
	  BX_VGA_THIS write(0xfd96,(((k>>5)&7)*0x49)>>1,1,0);
	}
	break;

      case 16:
	for(addr=j*0x40000;
	    addr<j*0x40000+(0x80000/regsets[i].numlayers);
	    addr+=2) {
	  *((Bit16u*)(&BX_VGA_THIS s.video_memory[addr])) =
	    (j==0 ? addr : (~addr)) ;
	}
	break;

      default:
	printf("not found graphics mode.\n");
      }
    }

    if(regsets[i].numlayers==1) {
      printf("disp page 0.\n");
      BX_VGA_THIS write(0xfda0,0x0c,1,0);
    } else {
      printf("disp page 0 + 1(layer 0 first).\n");
      BX_VGA_THIS write(0xfda0,0x0f,1,0);
      BX_VGA_THIS write(0x0448,1,1,0);
      BX_VGA_THIS write(0x044a,regsets[i].shifter[1],1,0);
    }
    BX_VGA_THIS s.all_updated = 1;
    BX_VGA_THIS s.dispmode_updated = 1;
    BX_VGA_THIS update();\
    UP();
    W();

    if(regsets[i].numlayers==2) {
      printf("disp page 0.\n");
      BX_VGA_THIS write(0xfda0,0x0c,1,0);
      UP();
      W();

      printf("disp page 1.\n");
      BX_VGA_THIS write(0xfda0,0x03,1,0);
      UP();
      W();

      printf("disp page 0 + 1(layer 1 first).\n");
      BX_VGA_THIS write(0xfda0,0x0f,1,0);
      BX_VGA_THIS write(0x0448,1,1,0);
      BX_VGA_THIS write(0x044a,regsets[i].shifter[1] | 0x01,1,0);
      UP();
      W();
    }

    printf("disp stop.\n");
    BX_VGA_THIS write(0xfda0,0x00,1,0);
    UP();
    W();

    if(regsets[i].numlayers==1) {
      printf("disp page 0.\n");
      BX_VGA_THIS write(0xfda0,0x0c,1,0);
    } else {
      printf("disp page 0 + 1(layer 0 first).\n");
      BX_VGA_THIS write(0xfda0,0x0f,1,0);
      BX_VGA_THIS write(0x0448,1,1,0);
      BX_VGA_THIS write(0x044a,regsets[i].shifter[1],1,0);
    }
    UP();

    printf("benchmark");
    gettimeofday(&start,NULL);
    const int numframe=100;
    for(j=0;j<numframe;j+=2) {
      BX_VGA_THIS write(0xfda0,0x00,1,0);
      BX_VGA_THIS update();
      bx_gui.flush();
      if(regsets[i].numlayers==1) {
	BX_VGA_THIS write(0xfda0,0x0c,1,0);
      } else {
	BX_VGA_THIS write(0xfda0,0x0f,1,0);
      }
      BX_VGA_THIS update();
      bx_gui.flush();
    }
    gettimeofday(&end,NULL);
    printf("%f(frame/s).\n",numframe/((end.tv_sec-start.tv_sec)+(end.tv_usec-start.tv_usec)/1000000.0));

    printf("screen clear.\n");
    for(j=0;j<regsets[i].numlayers;j++) {
      switch(modes[regsets[i].modes[j]].depth) {
      case 4:
      case 8:
	for(addr=j*0x40000;
	    addr<j*0x40000+(0x80000/regsets[i].numlayers);
	    addr++) {
	  BX_VGA_THIS mem_update(addr,0);
	  if(rand()<RAND_MAX/500) BX_VGA_THIS update();
	}
	break;

      case 16:
	for(addr=j*0x40000;
	    addr<j*0x40000+(0x80000/regsets[i].numlayers);
	    addr+=2) {
	  BX_VGA_THIS mem_update(addr  ,0x00);
	  BX_VGA_THIS mem_update(addr+1,0x80);
	  if(rand()<RAND_MAX/500) BX_VGA_THIS update();
	}
	break;

      default:
	printf("not found graphics mode.\n");
      }
    }
    UP();
    W();

    i++;
  }
}
#endif
#endif
